"""
Initialize an inventory
=======================

.. sectionauthor:: Julian Heinze (Helmholtz Centre for Environmental Research GmbH - UFZ)

This example shows four ways to initialise an inventory object using the Spatiocoexistence package.
The first method involves defining the number of different species and dimensions.
Secondly, by loading a file.
Thirdly, by defining a structured NumPy array.
Fourth, by copying the first inventory.
"""

# %%
# Necessary imports
from spatiocoexistence import Inventory
import numpy as np
from spatiocoexistence.data import inventory_file

# %%
# 1. Create a new random inventory from scratch. Individuals are placed according to the density of BCI randomly in the defined area.
# Also plot the inventory map afterwards.
inv1 = Inventory.from_random(n_species=5, dim_x=100, dim_y=100)
print(
    f"Random inventory: {inv1.data.shape[0]} individuals, fields: {inv1.data.dtype.names}"
)
inv1.plot_map()
# %%
# 2. Load an inventory from a file. Exchange 'inventory_file' with the path to your file.
# Also plot the inventory map afterwards.
inv2 = Inventory.from_data(inventory_file)
print(
    f"Loaded inventory: {inv2.data.shape[0]} individuals, fields: {inv2.data.dtype.names}"
)
inv2.plot_map(scale=5, filter=2)
# %%
# 3. Initialize from a structured numpy array (must match expected dtype)
dtype = inv1.data.dtype
n_individuals = 2000
arr = np.zeros(n_individuals, dtype=dtype)
arr["x"] = np.linspace(0, 10, n_individuals)
arr["y"] = np.linspace(0, 10, n_individuals)
arr["dbh"] = np.random.uniform(1, 10, n_individuals)
arr["species"] = np.arange(n_individuals)
arr["status"] = np.ones(n_individuals)
inv3 = Inventory.from_data(arr)
print(
    f"Inventory from array: {inv3.data.shape[0]} individuals, fields: {inv3.data.dtype.names}"
)

# %%
# 4. Copy from another Inventory object
inv4 = Inventory.from_data(inv1)
print(
    f"Copied inventory: {inv4.data.shape[0]} individuals, fields: {inv4.data.dtype.names}"
)
